
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_line.c */


#include "fs_itype.h"

#if defined(FS_BITMAPS)

#include "fs_bitmap.h"

/****************************************************************/
/* we do two passes (horizontal then vertical) when dropouts are ON */
FS_VOID line_dropout(_DS_ FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1)
{
    TLIST *temp;

    line_simple(_PS_ x0, y0, x1, y1);
    if (STATE.error)
        return;
    temp = (TLIST *)(STATE.server->tlist);

    STATE.server->tlist = STATE.server->drop;
    line_simple(_PS_ y0, x0, y1, x1);
    if (STATE.error)
    {
        STATE.server->tlist = temp;
        return;
    }
    STATE.server->tlist = temp;
}

/****************************************************************/
FS_VOID line_simple(_DS_ FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1)
{
    FS_FIXED y_min, y_max, x, dx;
    register int n;
    int up = 1;
    FS_SHORT y;
    TLIST *tlist;
    FS_FIXED hph_firstz = 0;
    FS_FIXED hph_lastz = 0;
    FS_SHORT hph_firsty = 0;
    FS_SHORT hph_lasty = 0;

    /* ignore horizontal lines */
    if (y1 == y0)
        return;

    /* make line segment point up */
    if (y1 < y0)
    {
        up = 0;
        x = y1;
        y1 = y0;
        y0 = x;
        x = x1;
        x1 = x0;
        x0 = x;
    }

    /* starting and ending half-integers */
    y_min = ABOVE(y0);
    y_max = BELOW(y1);
    if (y_max < y_min)
        return;

    /* get ducks in a row */
    y = FS_FLOOR(y_min);
    n = FS_FLOOR(y_max) - y + 1;
    tlist = (TLIST *)(STATE.server->tlist);

    /* special case vertical lines */
    if (x0 == x1)
    {
        FS_FIXED z = up | (x0 << 1);
        hph_firstz = z;
        hph_firsty = y;
        while (n--)
        {
            append_tlist(_PS_ z, y, tlist);
            if (STATE.error)
                return;
            y++;
        }
        hph_lastz = z;
        hph_lasty = y - 1 ;
    }
    else /* other slopes */
    {
        if (n == 1)
        {
            x = x0 + LongMulDiv(y_min - y0, x1 - x0, y1 - y0);
            hph_firstz = hph_lastz = up | (x << 1);
            hph_firsty = hph_lasty = y;
            append_tlist(_PS_ hph_firstz, y, tlist);
            if (STATE.error)
                return;
        }
        else
        {
            dx = FixDiv(x1 - x0, y1 - y0);
            x = x0 + FixMul(y_min - y0, dx);
            dx <<= 1;
            x = (x << 1) | up;
            hph_firstz = x;
            hph_firsty = y;
            while (n--)
            {
                append_tlist(_PS_ x, y, tlist);
                if (STATE.error)
                    return;
                x += dx;
                y++;
            }

            hph_lastz = x - dx;
            hph_lasty = y - 1 ;
        }
    }

    if (up)
    {
        if (tlist->hph_flag == 0)
        {
            tlist->hph_loopstartz = hph_firstz;
            tlist->hph_loopstarty = hph_firsty;
            tlist->hph_prevz = hph_lastz;
            tlist->hph_prevy = hph_lasty;
            tlist->hph_flag = 1;
        }
        else if (tlist->hph_flag == 2)
        {
            tlist->hph_flag = 3;
            tlist->hph_prevz = hph_lastz;
            tlist->hph_prevy = hph_lasty;
        }
        else if ((tlist->hph_prevz & 1) != (hph_firstz & 1) && tlist->hph_prevy == hph_firsty &&
                 FS_ROUND(tlist->hph_prevz >> 1) == FS_ROUND(hph_firstz >> 1))
        {
            remove_tran(tlist->hph_prevz, tlist->hph_prevy, tlist);
            remove_tran(hph_firstz, hph_firsty, tlist);
            if (tlist->hph_flag == 1)
                tlist->hph_loopstarty = tlist->max + 1;

            tlist->hph_flag = 2;
        }
        else
        {
            tlist->hph_flag = 3;
            tlist->hph_prevz = hph_lastz;
            tlist->hph_prevy = hph_lasty;
        }
    }
    else
    {
        if (tlist->hph_flag == 0)
        {
            tlist->hph_loopstartz = hph_lastz;
            tlist->hph_loopstarty = hph_lasty;
            tlist->hph_prevz = hph_firstz;
            tlist->hph_prevy = hph_firsty;
            tlist->hph_flag = 1;
        }
        else if (tlist->hph_flag == 2)
        {
            tlist->hph_flag = 3;
            tlist->hph_prevz = hph_firstz;
            tlist->hph_prevy = hph_firsty;
        }
        else if ( (tlist->hph_prevz & 1) != (hph_lastz & 1) &&
                  tlist->hph_prevy == hph_lasty &&
                  FS_ROUND(tlist->hph_prevz >> 1) == FS_ROUND(hph_lastz >> 1))
        {
            remove_tran(tlist->hph_prevz, tlist->hph_prevy, tlist);
            remove_tran(hph_lastz, hph_lasty, tlist);
            if (tlist->hph_flag == 1)
                tlist->hph_loopstarty = tlist->max + 1;
            tlist->hph_flag = 2;
        }
        else
        {
            tlist->hph_prevz = hph_firstz;
            tlist->hph_prevy = hph_firsty;
            tlist->hph_flag = 3;
        }
    }
}

/****************************************************************/
/* recursively divide the quadratic into 2^levels line segments */
static FS_VOID quad_divide(_DS_ FS_SHORT levels, FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2)
{
    FS_FIXED lx, ly, rx, ry, mx, my;

    /* subdivide */
    lx = (x0 + x1) >> 1;
    ly = (y0 + y1) >> 1;
    rx = (x1 + x2) >> 1;
    ry = (y1 + y2) >> 1;
    mx = (lx + rx) >> 1;
    my = (ly + ry) >> 1;

    if (levels <= 1)
    {
        (*STATE.server->line)(_PS_ x0, y0, mx, my);
        (*STATE.server->line)(_PS_ mx, my, x2, y2);
        return;
    }
    else
    {
        levels--;
        quad_divide(_PS_ levels, x0, y0, lx, ly, mx, my);
        quad_divide(_PS_ levels, mx, my, rx, ry, x2, y2);
    }
}

/****************************************************************/
/* using a flatness which depends on character size */
/* computes a polygon approximation to a quadratic  */
FS_VOID FS_quad(_DS_ FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2)
{
    FS_FIXED mx, my, d, err;
    FS_SHORT levels;

    /* Is the control point (x1,y1) within 1/1024 of an EM of chord midpoint (mx,my)

                         ___---x1y1
                   ___---          \
             ___---                 \
       ___---                        \
    x0y0--------------+--------------x2y2
                    mx,xy

    err = lpm / 1024.0
    at 13 lpm it's .0127 < .015625 == 1/64 pixel tolerance
    at 100 lpm it's about 1/10 pixel tolerance
    at 1000 lpm it's about 1 pixel tolerance
    */

    mx = (1 + x0 + x2) >> 1;
    my = (1 + y0 + y2) >> 1;
    d = DistanceNorm(x1 - mx, y1 - my);
    err = (FS_FIXED)STATE.lpm << 6;

    if (d <= err)
        (*STATE.server->line)(_PS_ x0, y0, x2, y2);
    else
    {
        /* how many levels of bisection are needed */
        levels = 0;
        while (d >= err)
        {
            levels++;
            d >>= 2;
        }
        quad_divide(_PS_ levels, x0, y0, x1, y1, x2, y2);
    }
}

/****************************************************************/
/* actually divide the cubic into 2^^levels line segments */
static FS_VOID cube_divide(_DS_ FS_SHORT levels, FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2, FS_FIXED x3, FS_FIXED y3)
{
    FS_FIXED ax, bx, cx, dx, ex, mx, ay, by, cy, dy, ey, my;

    ax = (x0 + x1) >> 1;
    bx = (x1 + x2) >> 1;
    cx = (x2 + x3) >> 1;
    dx = (ax + bx) >> 1;
    ex = (bx + cx) >> 1;
    mx = (dx + ex) >> 1;
    ay = (y0 + y1) >> 1;
    by = (y1 + y2) >> 1;
    cy = (y2 + y3) >> 1;
    dy = (ay + by) >> 1;
    ey = (by + cy) >> 1;
    my = (dy + ey) >> 1;

    if (levels == 1)
    {
        (*STATE.server->line)(_PS_ x0, y0, mx, my);
        (*STATE.server->line)(_PS_ mx, my, x3, y3);
    }
    else
    {
        levels--;
        cube_divide(_PS_ levels, x0, y0, ax, ay, dx, dy, mx, my);
        cube_divide(_PS_ levels, mx, my, ex, ey, cx, cy, x3, y3);
    }
}

/****************************************************************/
/* using a flatness which depends on character size */
/* computes a polygon approximation to a cubic  */
FS_VOID FS_cube(_DS_ FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2, FS_FIXED x3, FS_FIXED y3)
{
    FS_FIXED xL, yL, xR, yR, dL, dR, d, err;
    FS_SHORT levels;

    /*
    Is the control point (x1,y1) within 1/1024 of an EM of (xL,yL)
    AND is (x2,y2) within 1/1024 of an EM of (xR,yR) ??

             x1,y1 -------- x2,y2
           /                   \
         /                         \
       /                           \
    x0,y0------+-----------+--------x3,y3
            xL,yL        xR,yR

    err = lpm / 1024.0
    at 13 lpm it's .0127 < .015625 == 1/64 pixel tolerance
    at 100 lpm it's about 1/10 pixel tolerance
    at 1000 lpm it's about 1 pixel tolerance
    */

    xL = x0 + (x3 - x0) / 3;
    yL = y0 + (y3 - y0) / 3;
    dL = DistanceNorm(x1 - xL, y1 - yL);

    xR = x0 + ((x3 - x0) << 1) / 3;
    yR = y0 + ((y3 - y0) << 1) / 3;
    dR = DistanceNorm(x2 - xR, y2 - yR);

    d = MAX(dL, dR);
    err = (FS_FIXED)STATE.lpm << 6;

    if (d <= err)
        (*STATE.server->line)(_PS_ x0, y0, x3, y3);
    else
    {
        /* how many levels of bisection are needed */
        levels = 0;
        while (d > err)
        {
            levels++;
            d >>= 2;
        }
        cube_divide(_PS_ levels, x0, y0, x1, y1, x2, y2, x3, y3);
    }
}

/****************************************************************/
#endif /* if defined(FS_BITMAPS) || defined(FS_GRAYMAPS)
*/
